//
//  ViewController.swift
//  PlayerPluginExample
//
//  Created by Ole Helgesen on 11/03/2019.
//  Copyright © 2019 Ease Live AS. All rights reserved.
//

import UIKit
import AVFoundation
import AVKit
import EaseLiveSDK

class ViewController: UIViewController {
    var playerPlugin: MyPlayerPlugin?
    var player: AVPlayer?
    var easeLive: EaseLive?
    var playerObservers: AVPlayerObservers?
    
    // custom player controls
    var customPlayerView: CustomPlayerView?
    
    // default player controls
    var avPlayerViewController: AVPlayerViewController?
    
    // container for EaseLive overlay content
    var easeLiveView: UIView?
    
    // producer set overlay enabled, received in onEaseLiveAppStatus
    var easeLiveEnabled = true
    // user set overlay enabled, toggled by button in player controls. set false to hide overlay until enabled by user
    var easeLiveEnabledByUser = true
    
    // demo stream
    
    // VOD stream containing time in EXT-X-PROGRAM-DATE-TIME
    // let streamUrl = "https://s3-eu-west-1.amazonaws.com/vod-assets-ireland-sixty-no/dev/hls_vod_with_id3_and_programtime_nosound/stream.m3u8"
    
    // live stream containing time in ID3 tags
    let streamUrl = "https://eu-dev.stream.easelive.tv/fotball/ngrp:Stream1_all/playlist.m3u8?DVR"
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(didEnterBackground),
                                               name: UIApplication.didEnterBackgroundNotification,
                                               object: nil)
        
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(willEnterForeground),
                                               name: UIApplication.willEnterForegroundNotification,
                                               object: nil)
        
        // set .mixWithOthers option of audio session if you want player and EaseLive to be able to play sound at the same time.
        // This allows video layers or watch party inside the EaseLive overlay to mix sound with your AVPlayer's sound
        do {
            let audioSession = AVAudioSession.sharedInstance()
            try audioSession.setCategory(.playback, mode: .moviePlayback, options: [.mixWithOthers])
            try audioSession.setActive(true)
            
        } catch {
            print("Setting category to AVAudioSessionCategoryPlayback failed.", error)
        }
        
        if let stream = URL(string: streamUrl) {
            // player setup
            let playerItem = AVPlayerItem(url: stream)
            let player = AVPlayer(playerItem: playerItem)
            self.player = player
            
            var useCustomPlayerView = true
            
            // use AVPlayerViewController's default player controls on iOS 16+, visionOS and tvOS
            #if os(tvOS)
            useCustomPlayerView = false
            #else
            if #available(iOS 16, visionOS 1, *) {
                useCustomPlayerView = false
            }
            #endif
            
            if useCustomPlayerView {
                // use custom player controls defined in CustomPlayerView
                setupCustomPlayerView()
            } else {
                // use default player controls of AVPlayerViewController
                setupDefaultPlayerView()
            }
            
            player.play()
            
            // EASELIVE setup
            
#if DEBUG
            // enables output of logs in the debug build
            EaseLive.setDebugging(debug: true)
#endif
            
            let playerPlugin = MyPlayerPlugin(player: player)
            playerObservers = AVPlayerObservers(player: player, playerPlugin: playerPlugin)
            
            playerPlugin.changePlayerControlsVisibilityCallback = { [weak self] visible in
                guard let self = self else { return }
                
                // change custom controls visibility
                if let playerView = self.customPlayerView {
                    if visible {
                        playerView.playbackControlView?.show()
                    } else {
                        playerView.playbackControlView?.hide()
                    }
                }
                
                // change default controls visibility
                if let avPlayerViewController {
                    if visible {
                        #if os(tvOS)
                        // no function to show player controls in AVPlayerViewController
                        // workaround to show player controls by pausing for a frame
                        if avPlayerViewController.player?.timeControlStatus == .playing {
                            avPlayerViewController.player?.pause()
                            avPlayerViewController.player?.play()
                        } else if avPlayerViewController.player?.timeControlStatus == .paused {
                            avPlayerViewController.player?.play()
                            avPlayerViewController.player?.pause()
                        }
                        #endif
                    }
                }
            }
            
            playerPlugin.changePlayerVideoScaleCallback = { [weak self] scaleX, scaleY, pivotX, pivotY in
                guard let self = self else { return }
                
                // scale video in custom player
                if let customPlayerView {
                    customPlayerView.playerView.setVideoScale(scaleX: scaleX, scaleY: scaleY, pivotX: pivotX, pivotY: pivotY)
                }
                // scale video in AVPlayerViewController
                else if let avPlayerViewController {
                    avPlayerViewController.setVideoScale(scaleX: scaleX, scaleY: scaleY, pivotX: pivotX, pivotY: pivotY)
                }
            }
            self.playerPlugin = playerPlugin
            
            // register for notifications from the SDK
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveError(notification:)),
                                                   name: EaseLiveNotificationKeys.easeLiveError,
                                                   object: nil)
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveReady(notification:)),
                                                   name: EaseLiveNotificationKeys.easeLiveReady,
                                                   object: nil)
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveAppStatus(notification:)),
                                                   name: EaseLiveNotificationKeys.bridgeAppStatus,
                                                   object: nil)
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveBridgeMessage(notification:)),
                                                   name: EaseLiveNotificationKeys.bridgeMessage,
                                                   object: nil)
            
            // The overlay will be added as a subview to the view that is passed to the EaseLive constructor.
            // This view should be layered above the player video surface and below the player controls, and must be able to receive touch events.
            
            let accountId: String
            let projectId: String?
            let programId: String
            let env: String
            var params: [String: Any] = [:]
            #if os(tvOS)
                accountId = "tutorials"
                projectId = "580e5ba1-423d-4c17-9a86-08b75e88bb7e"
                programId = "cloud-capture-demo"
                env = "prod"
            #else
                accountId = "tutorials"
                projectId = "0346ae3e-7a91-4760-bcd3-cd84bb6790dd"
                programId = "2d2711ff-6ff2-41c1-a141-060e9ffa2c38"
                env = "prod"
            
                if avPlayerViewController != nil {
                    // AVPlayerViewController on iOS and visionOS:
                    // Sets parameter to enable passthrough of touches on the overlay background.
                    // This is done because AVPlayerViewController does not have functionality to toggle the player controls visibility or know the current visibility, so the touches needs to be handled by the native view.
                    params["stageTouchPassthrough"] = true
                }
            #endif
            
            self.easeLive = EaseLive(parentView: easeLiveView!,
                                     accountId: accountId,
                                     projectId: projectId,
                                     programId: programId,
                                     env: env,
                                     params: params,
                                     playerPlugin: playerPlugin)
            
            // before calling easeLive.create() you can register other plugins using easeLive.use()
            self.easeLive?.create()
            
            // a call to onReady() on the player plugin is required to let the SDK know the player is ready, so it will know to proceed with loading the overlay UI.
            // this could be used to wait to load the overlay UI until the video is loaded
            self.playerPlugin?.onReady()
        }
    }
    
    override func viewLayoutMarginsDidChange() {
        
        super.viewLayoutMarginsDidChange()
        
        if avPlayerViewController != nil {
            // reapply AVPlayerViewController videoscale after rotation
            UIView.animate(withDuration: TimeInterval(0.1), animations: {
                let l = self.avPlayerViewController?.findPlayerView(parentView: self.view)?.layer
                l?.transform = CATransform3DMakeScale(1, 1, 0)
            }) { finished in
                self.playerPlugin?.updateVideoScale()
            }
        }
    }
    
    @objc func didEnterBackground() {
        self.easeLive?.pause()
    }

    @objc func willEnterForeground() {
        self.easeLive?.load()
    }
    
    // setup a player using custom controls
    func setupCustomPlayerView() {
        let playerView = CustomPlayerView(frame: .zero)
        playerView.translatesAutoresizingMaskIntoConstraints = false
        playerView.player = player
        playerView.onPlayerControlsVisibilityChanged = { [weak self] visible in
            self?.playerPlugin?.onControllerVisibilityChanged(visible: visible)
        }
        self.customPlayerView = playerView
        view.addSubview(playerView)
        
        NSLayoutConstraint.activate([
            playerView.leftAnchor.constraint(equalTo: view.leftAnchor),
            playerView.topAnchor.constraint(equalTo: view.topAnchor),
            playerView.rightAnchor.constraint(equalTo: view.rightAnchor),
            playerView.bottomAnchor.constraint(equalTo: view.bottomAnchor)
        ])
        
        easeLiveView = playerView.overlayView
        
        playerView.playbackControlView.show()
    }
    
    // setup a player using default AVPlayerViewController controls
    func setupDefaultPlayerView() {
        
        let easeLiveView = UIView(frame: .zero)
        easeLiveView.translatesAutoresizingMaskIntoConstraints = false
        
        let avpvc = AVPlayerViewController()
        avpvc.delegate = self
        avPlayerViewController = avpvc
        avpvc.showsPlaybackControls = true
        avpvc.allowsPictureInPicturePlayback = true
        #if os(iOS)
            avpvc.canStartPictureInPictureAutomaticallyFromInline = true
        #endif
        avpvc.player = player

        #if os(tvOS)
        if #available(tvOS 15.0, *) {
            avpvc.transportBarIncludesTitleView = true
            avpvc.playbackControlsIncludeTransportBar = true
            
            // example of a button to let the user toggle the overlay visibility
            let toggleOverlayAction = UIAction(title: "Toggle overlay", image: easeLiveEnabledByUser ? .remove : .add) { [weak self] action in
                guard let self = self else { return }
                
                if easeLiveEnabled {
                    easeLiveEnabledByUser = !easeLiveEnabledByUser
                    self.easeLiveView?.isHidden = false
                    action.image = easeLiveEnabledByUser ? .remove : .add
                    
                    UIView.animate(withDuration: 0.2) {
                        self.easeLiveView?.alpha = self.easeLiveEnabledByUser ? 1 : 0
                        self.setNeedsFocusUpdate()
                    }
                }
            }

            avpvc.transportBarCustomMenuItems = [toggleOverlayAction]
        }
        #endif
        
        avpvc.view.frame = view.bounds
        avpvc.beginAppearanceTransition(true, animated: true)
        addChild(avpvc)
        view.addSubview(avpvc.view)
        avpvc.didMove(toParent: self)
        avpvc.endAppearanceTransition()
        
        #if os(tvOS)
            // NOTE: for use with AVPlayerViewController, the EaseLive parent view must be added in front of the AVPlayerViewController's view, or EaseLive cannot get focus, because the player controls cover the whole screen
            view.addSubview(easeLiveView)
            easeLiveView.superview?.bringSubviewToFront(easeLiveView)
            
            NSLayoutConstraint.activate([
                easeLiveView.leftAnchor.constraint(equalTo: view.leftAnchor),
                easeLiveView.topAnchor.constraint(equalTo: view.topAnchor),
                easeLiveView.rightAnchor.constraint(equalTo: view.rightAnchor),
                easeLiveView.bottomAnchor.constraint(equalTo: view.bottomAnchor)
            ])
        #else
            if #available(iOS 16, visionOS 1, *) {
                // iOS 16+ and visionOS, show EL in AVPlayerViewController's contentOverlayView
                if let view = avpvc.contentOverlayView {
                    view.addSubview(easeLiveView)
                    easeLiveView.leftAnchor.constraint(equalTo: view.leftAnchor).isActive = true
                    easeLiveView.topAnchor.constraint(equalTo: view.topAnchor).isActive = true
                    easeLiveView.rightAnchor.constraint(equalTo: view.rightAnchor).isActive = true
                    easeLiveView.bottomAnchor.constraint(equalTo: view.bottomAnchor).isActive = true
                }
            } else {
                // In AVPlayerViewController on iOS 15 and older contentOverlayView is not interactive, so there isn't any good solution to insert the EaseLive overlay between the video surface and the player controls of AVPlayerViewController.
                // instead use custom player controls or only show EaseLive on iOS 16 and newer
            }
        #endif
        
        self.easeLiveView = easeLiveView
        easeLiveView.isHidden = !easeLiveEnabled
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        navigationController?.isNavigationBarHidden = true
        player?.play()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        player?.pause()
        
        if isBeingDismissed || isMovingFromParent || navigationController?.isBeingDismissed ?? false {
            // destroy EaseLive instance when it will stop being used
            easeLive?.destroy()
            NotificationCenter.default.removeObserver(self)
        }
    }
    
    // on tvOS configure when EL and when the player should have focus. In this example EL is hidden while player controls are visible, see willTransitionToVisibilityOfTransportBar
    override var preferredFocusEnvironments: [UIFocusEnvironment] {
        if let customPlayerView, customPlayerView.playbackControlView.visible {
            print("focus custom player controls")
            return [customPlayerView.playbackControlView]
        }
        
        if let easeLiveView, let easeLive, !easeLiveView.isHidden && easeLiveView.alpha != 0 && !easeLive.preferredFocusEnvironments.isEmpty {
            print("focus EL")
            return easeLive.preferredFocusEnvironments
        }
        
        if let avPlayerViewController {
            print("focus AVPlayerViewController")
            return avPlayerViewController.preferredFocusEnvironments
        }
        
        print("focus app")
        return super.preferredFocusEnvironments
    }
    
    // notification that the overlay successfully loaded
    @objc func onEaseLiveReady(notification: Notification) {
        print("EaseLive ready")
        setNeedsFocusUpdate()
    }
    
    // notification that the producer changed status of the loaded overlay.
    // When the overlay was disabled, it should be removed
    @objc func onEaseLiveAppStatus(notification: Notification) {
        if let status = notification.userInfo?[EaseLiveNotificationKeys.statusUserInfoKey] as? String {
            print("EaseLive status: \(status)")
            if status == "disabled" {
                easeLiveEnabled = false
                easeLive?.destroy()
                easeLive = nil
                easeLiveView?.isHidden = true
            }
            else if status == "hidden" {
                easeLiveEnabled = false
                easeLiveView?.isHidden = true
            }
            else if status == "enabled" {
                easeLiveEnabled = true
                easeLiveView?.isHidden = false
            }
            setNeedsFocusUpdate()
            updateFocusIfNeeded()
        }
    }
    
    // notification that an error occurred
    @objc func onEaseLiveError(notification: Notification) {
        if let error = notification.userInfo?[EaseLiveNotificationKeys.errorUserInfoKey] as? EaseLiveError {
            if error.level == .fatal {
                // fatal error. For example the UI failed to load.
                // remove the overlay and fallback to a normal video
                easeLive?.destroy()
                easeLive = nil
                easeLiveView?.isHidden = true
            } else {
                // non-fatal error/warning.
            }
        }
    }
    
    // custom message from the overlay UI
    @objc func onEaseLiveBridgeMessage(notification: Notification) {
        do {
            guard let jsonString = notification.userInfo?[EaseLiveNotificationKeys.jsonStringUserInfoKey] as? String else { return }
            guard let json = try JSONSerialization.jsonObject(with: Data(jsonString.utf8)) as? [String: Any] else { return }
            guard let event = json["event"] as? String, let metadata = json["metadata"] as? [String: Any] else { return }
            
            // example of a message used for sharing a Watch Party invitation
            if event == "inviteMessage", let message = metadata["message"] as? String {
#if os(iOS)
                let objectsToShare = [message] as [Any]
                let activityVC = UIActivityViewController(activityItems: objectsToShare, applicationActivities: nil)
                activityVC.excludedActivityTypes = [.airDrop, .addToReadingList]
                self.present(activityVC, animated: true) {}
#endif
            }
        } catch {
        }
    }
}


extension ViewController: @preconcurrency AVPlayerViewControllerDelegate {
#if os(tvOS)
    // listen for AVPlayerViewController's controls visibility on tvOS
    func playerViewController(_ playerViewController: AVPlayerViewController, willTransitionToVisibilityOfTransportBar visible: Bool, with coordinator: AVPlayerViewControllerAnimationCoordinator) {
        // when player controls hide, move focus back to EaseLive
        if !visible {
            if easeLive != nil && easeLiveEnabled && easeLiveEnabledByUser {
                UIView.animate(withDuration: 0.2) {
                    self.easeLiveView?.alpha = 1
                    self.setNeedsFocusUpdate()
                }
            }
        } else {
            UIView.animate(withDuration: 0.2) {
                self.easeLiveView?.alpha = 0
                self.setNeedsFocusUpdate()
            }
        }
        
        playerPlugin?.onControllerVisibilityChanged(visible: visible)
    }
#endif
    
    func playerViewControllerWillStartPictureInPicture(_ playerViewController: AVPlayerViewController) {
        easeLiveView?.isHidden = true
    }
    
    func playerViewControllerDidStopPictureInPicture(_ playerViewController: AVPlayerViewController) {
        easeLiveView?.isHidden = !easeLiveEnabled
    }
}

extension AVPlayerViewController {
    // find the view that has the video rendering layer
    func findPlayerView(parentView: UIView) -> UIView? {
        for v in parentView.subviews {
            if v.layer as? AVPlayerLayer != nil {
                return v
            } else {
                return findPlayerView(parentView: v)
            }
        }
        return nil
    }
    
    // scale player layer
    func setVideoScale(scaleX: Float, scaleY: Float, pivotX: Float, pivotY: Float) {
        if let avpcView = self.viewIfLoaded,
           let playerView = findPlayerView(parentView: avpcView),
           let playerLayer = playerView.layer as? AVPlayerLayer {
            let videoBounds = playerLayer.videoRect
            let defaultAnchorPoint = CGPoint(x: 0.5, y: 0.5)
            let anchorPoint = CGPoint(x: CGFloat(pivotX), y: CGFloat(pivotY))
            var shiftX = CGFloat(0)
            var shiftY = CGFloat(0)
            if anchorPoint.x != defaultAnchorPoint.x {
                shiftX = videoBounds.width * (anchorPoint.x - defaultAnchorPoint.x)
                
                let scaledW = videoBounds.width * CGFloat(scaleX)
                if anchorPoint.x > 0.5 {
                    shiftX -= scaledW * 0.5
                } else {
                    shiftX += scaledW * 0.5
                }
            }
            if anchorPoint.y != defaultAnchorPoint.y {
                shiftY = videoBounds.height * (anchorPoint.y - defaultAnchorPoint.y)
                
                let scaledH = videoBounds.height * CGFloat(scaleY)
                if anchorPoint.y > 0.5 {
                    shiftY -= scaledH * 0.5
                } else {
                    shiftY += scaledH * 0.5
                }
            }
            var t = CATransform3DIdentity
            t = CATransform3DTranslate(t, shiftX, shiftY, 0)
            t = CATransform3DScale(t, CGFloat(scaleX), CGFloat(scaleY), 0)
            playerLayer.transform = t
        }
    }
}
