//
//  PlayerObservers.swift
//  PlayerPluginExample
//
//  Created by Ole Helgesen on 24/11/2025.
//  Copyright © 2025 Ease Live. All rights reserved.
//
import UIKit
import AVFoundation
import AVKit
import EaseLiveSDK

extension AVPlayerItem {
    // get correct duration of live stream with seekable buffer
    func playerDuration() -> Float64 {
        var duration: Float64 = 0
        if status == .readyToPlay, let range = seekableTimeRanges.last {
            let rangeDuration = CMTimeGetSeconds(range.timeRangeValue.duration)
            
            if !rangeDuration.isNaN && !rangeDuration.isInfinite {
                duration = rangeDuration
            }
        }
        return duration
    }
}

@MainActor final class AVPlayerObservers: NSObject, AVPlayerItemMetadataOutputPushDelegate {
    weak var playerPlugin: MyPlayerPlugin?
    weak var player: AVPlayer?
    
    var timedMetadataQueue = DispatchQueue(label: "TimedMetadataQueue", qos: .background)
    var timedMetadataOutput = AVPlayerItemMetadataOutput()
    
    var timeControlObserver: NSKeyValueObservation?
    var statusObserver: NSKeyValueObservation?
    var rateObserver: NSKeyValueObservation?
    var volumeObserver: NSKeyValueObservation?
    var mutedObserver: NSKeyValueObservation?
    
    var hasId3 = false
    
    // listen for changes in AVPlayer and send the change to EaseLive
    init(player: AVPlayer, playerPlugin: MyPlayerPlugin) {
        self.player = player
        self.playerPlugin = playerPlugin
        super.init()

        // set up observers for change in player state and metadata
        self.statusObserver = player.observe(\.currentItem?.status, options: [.initial, .new], changeHandler: { [weak self] (player, change) in
            if let itemStatus = player.currentItem?.status {
                Task { [weak self] in
                    await self?.onStatusChanged(itemStatus: itemStatus)
                }
            }
        })
        
        self.timeControlObserver = player.observe(\.timeControlStatus, options: [.initial, .new], changeHandler: { [weak self] player, change in
            let status = player.timeControlStatus
            Task { [weak self] in
                await self?.onTimeControlStatusChanged(status: status)
            }
        })
        
        self.rateObserver = player.observe(\.rate, options: [.initial, .new], changeHandler: { [weak self] (player, change) in
            Task { [weak self] in
                await self?.onRateChanged(rate: player.rate)
            }
        })
        
        self.volumeObserver = player.observe(\.volume, changeHandler: { [weak self] player, change in
            Task { [weak self] in
                await self?.onVolumeChanged(volume: player.volume)
            }
        })
        
        self.mutedObserver = player.observe(\.isMuted) { [weak self] player, change in
            Task { [weak self] in
                await self?.onMutedChanged(isMuted: player.isMuted)
            }
        }
        
        // reads time from EXT-X-PROGRAM-DATE-TIME
        player.addPeriodicTimeObserver(forInterval: CMTime(seconds: 0.5, preferredTimescale: CMTimeScale(NSEC_PER_SEC)), queue: .main) { [weak self] (time) in
            Task { [weak self] in
                await self?.onPlayerTimeChanged()
            }
        }
        
        self.timedMetadataOutput.setDelegate(self, queue: timedMetadataQueue)
        
        player.currentItem?.add(timedMetadataOutput)
    }
    
    // sends the current audio volume to EaseLive
    @MainActor private func onVolumeChanged(volume: Float) {
        self.playerPlugin?.onVolumeChanged(volume: Int(floorf(volume * 100)))
    }
    
    // sends the current audio mute status to EaseLive
    @MainActor private func onMutedChanged(isMuted: Bool) {
        self.playerPlugin?.onMuteChanged(mute: isMuted)
    }
    
    // sends the current player state to EaseLive
    @MainActor private func onTimeControlStatusChanged(status: AVPlayer.TimeControlStatus) {
        if status == .playing {
            self.playerPlugin?.onState(state: .playing)
        } else if status == .paused {
            self.playerPlugin?.onState(state: .paused)
        } else if status == .waitingToPlayAtSpecifiedRate {
            self.playerPlugin?.onState(state: .buffering)
        }
    }
    
    // sends the current player speed to EaseLive
    @MainActor private func onRateChanged(rate: Float) {
        self.playerPlugin?.onSpeedChanged(speed: rate)
    }
    
    // send when the player is ready or received a fatal error
    @MainActor private func onStatusChanged(itemStatus: AVPlayerItem.Status) {
        switch itemStatus {
        case .failed:
            self.playerPlugin?.onError(error: EaseLiveError.playerConnectivity(level: .fatal, reason: "Failed to play stream"))
        default:
            print("unknown status for player")
        }
    }
    
    // send the absolute time of the current playback position. How this is done depends on how the time is inserted in the stream
    // NOTE: pick and use only one of the methods, either ID3 or EXT-X-PROGRAM-DATE-TIME. Both are here as an example. Using multiple time sources can lead to fluctuations
    
    // Listen for when the player reads time from HLS manifest tag EXT-X-PROGRAM-DATE-TIME
    @MainActor private func onPlayerTimeChanged() {
        guard let currentItem = self.player?.currentItem else { return }
        
        if !hasId3, let pdt = currentItem.currentDate() {
            // all values in UTC milliseconds
            let time = Int64(pdt.timeIntervalSince1970 * 1000)
            let position = Int64(CMTimeGetSeconds(currentItem.currentTime()) * 1000)
            let duration = Int64(currentItem.playerDuration() * 1000)
            self.playerPlugin?.onTime(time: time, position: position, duration: duration)
        }
    }
    
    // or read time from ID3 tags in the stream
    nonisolated public func metadataOutput(_ output: AVPlayerItemMetadataOutput, didOutputTimedMetadataGroups groups: [AVTimedMetadataGroup], from track: AVPlayerItemTrack?) {
        for group in groups {
            for item in group.items {
                if let string = item.stringValue, let utc = parseUtcFromId3Value(string) {
                    
                    let time = utc
                    Task { @MainActor [weak self] in
                        self?.hasId3 = true
                        self?.sendTime(time)
                    }
                }
            }
        }
    }
    
    @MainActor func sendTime(_ time: Int64) {
        guard let player else { return }
        guard let currentItem = player.currentItem else { return }
        
        let duration = Int64(currentItem.playerDuration() * 1000)
        
        let position = duration > 0 ? Int64(CMTimeGetSeconds(player.currentTime()) * 1000) : 0
        playerPlugin?.onTime(time: time, position: position, duration: duration)
    }
    
    // Parse ID3 metadata value to retrieve absolute time for the current frame and return it as UTC milliseconds
    nonisolated func parseUtcFromId3Value(_ value: String) -> Int64? {
        var utc: Int64?
        
        // NOTE: must modify to extract the correct metadata frame from your stream.
        // this is an example of how it is read from ID3 in EaseLive's test streams only.
        
        // example: JSON with the time as milliseconds, eg. {"ut":"1557311377000"}
        if let json = try? JSONSerialization.jsonObject(with: Data(value.utf8)) as? [String: Any] {
            if let utcInt = json["ut"] as? Int64 {
                utc = utcInt
            } else if let utcString = json["ut"] as? String, let utcInt = Int64(utcString) {
                utc = utcInt
            }
        }
        
        // example: string with the time as ISO 8601, eg. 2019-05-08T10:29:37.000Z
        if utc == nil {
            let formatter = ISO8601DateFormatter()
            formatter.formatOptions = [.withInternetDateTime, .withFractionalSeconds]
            if let date = formatter.date(from: value) {
                utc = Int64(date.timeIntervalSince1970 * 1000)
            }
        }
        
        return utc
    }
}
