//
//  MyPlayerPlugin.swift
//  PlayerPluginExample
//
//  Created by Ole Helgesen on 24/11/2025.
//  Copyright © 2025 Ease Live. All rights reserved.
//
import UIKit
import AVFoundation
import AVKit
import EaseLiveSDK

public class MyPlayerPlugin: PlayerPluginBase {
    weak var player: AVPlayer?
    
    var changePlayerControlsVisibilityCallback: ((_ visible: Bool) -> Void)?
    var changePlayerVideoScaleCallback: ((_ scaleX: Float, _ scaleY: Float, _ pivotX: Float, _ pivotY: Float) -> Void)?
    
    public init(player: AVPlayer) {
        super.init()
        self.TAG = String(describing: MyPlayerPlugin.self)
        self.player = player
    }
    
    // Called when the overlay UI sends an event to change the video URL.
    public override func playVideoUrl(url: String) {
        if let videoUrl = URL(string: url) {
            let currentItem = AVPlayerItem(url: videoUrl)
            player?.replaceCurrentItem(with: currentItem)
            player?.play()
        }
    }
    
    // Called when the overlay UI sends an event to change the player controls visibility.
    public override func setControllerVisible(visible: Bool) {
        changePlayerControlsVisibilityCallback?(visible)
    }
    
    // Called when a swipe gesture is detected on a part of the overlay where the video is visible
    // (ie outside any EaseLive graphical element). Can be used to trigger native UI.
    public override func didSwipeStage(direction: UISwipeGestureRecognizer.Direction) {
        if direction == .down {
            print("swiped down")
        }
    }
    
    // Called when the overlay UI sends an event to seek to a player position.
    public override func setTime(time: Int64) {
        // the time parameter is the UTC timecode for the wanted seek position.
        // PlayerPluginBase#currentTimecode is UTC that was last passed to PlayerPluginBase#onTime()
        guard let currentTimecode = self.currentTimecode else {
            return
        }
        
        // seek length is the difference between the wanted UTC and the current UTC
        let diff = Float64(time - currentTimecode)
        
        if let currentPositionTime = player?.currentTime()  {
            let currentPosition = CMTimeGetSeconds(currentPositionTime)
            
            // seek position relative to the current timeline position
            let position = currentPosition + diff
            
            let seekTime: CMTime = CMTime(seconds: Double(position) / 1000, preferredTimescale: 1000)
            let tolerance: CMTime = CMTime(seconds: Double(5), preferredTimescale: 1000)
            
            player?.seek(to: seekTime, toleranceBefore: tolerance, toleranceAfter: tolerance)
        }
    }
    
    // Called when the overlay UI sends an event to set the player state.
    public override func setState(state: PlayerState) {
        if state == .paused {
            player?.pause()
        } else if state == .playing {
            player?.play()
        }
    }
    
    // Called when the overlay UI sends an event to set the playback speed
    public override func setSpeed(speed: Float) {
        player?.rate = speed
    }
    
    // Called when the overlay UI sends an event to set the audio volume
    public override func setVolume(volume: Int) {
        player?.volume = Float(volume) / Float(100)
    }
    
    // Called when the overlay UI sends an event to set the audio mute status
    public override func setMute(mute: Bool) {
        player?.isMuted = mute
    }
    
    var scaleX: Float = 1
    var scaleY: Float = 1
    var pivotX: Float = 0
    var pivotY: Float = 0
    
    func updateVideoScale() {
        self.changePlayerVideoScaleCallback?(scaleX, scaleY, pivotX, pivotY)
    }
    
    // Called when the overlay UI sends an event to change the video scale and position
    public override func setVideoScale(scaleX: Float, scaleY: Float, pivotX: Float, pivotY: Float, duration: Int) {
        self.scaleX = scaleX
        self.scaleY = scaleY
        self.pivotX = pivotX
        self.pivotY = pivotY
        UIView.animate(withDuration: TimeInterval(Double(duration) / Double(1000)), animations: {
            self.updateVideoScale()
        })
        
        // notify EaseLive that the video will scale
        onVideoScale(scaleX: scaleX, scaleY: scaleY, pivotX: pivotX, pivotY: pivotY, duration: duration)
    }
}

