//
//  CustomPlayerView.swift
//  PlayerPluginExample
//
//  Created by Ole Helgesen on 15/02/2024.
//  Copyright © 2024 Ease Live. All rights reserved.
//

import UIKit
import AVKit

// a custom video player using AVPlayer with custom player controls


class CustomPlayerView: UIView {
    var player: AVPlayer? = nil {
        didSet {
            playerView.player = player
        }
    }
    var playerView: PlayerView!
    var overlayView: UIView!
    var playbackControlView: PlaybackControlView!
    
    
    var onPlayerControlsVisibilityChanged: ((_ visible: Bool) -> Void)?
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setup()
    }
    
    required init?(coder: NSCoder) {
        super.init(coder: coder)
        setup()
    }

    func setup() {
        playerView = PlayerView(frame: .zero)
        playerView.translatesAutoresizingMaskIntoConstraints = false
        addSubview(playerView)
        NSLayoutConstraint.activate([
            playerView.leftAnchor.constraint(equalTo: leftAnchor),
            playerView.topAnchor.constraint(equalTo: topAnchor),
            playerView.rightAnchor.constraint(equalTo: rightAnchor),
            playerView.bottomAnchor.constraint(equalTo: bottomAnchor)
        ])
        
        playerView.player = player
        
        overlayView = UIView(frame: .zero)
        overlayView.backgroundColor = .clear
        overlayView.translatesAutoresizingMaskIntoConstraints = false
        addSubview(overlayView)
        NSLayoutConstraint.activate([
            overlayView.leftAnchor.constraint(equalTo: leftAnchor),
            overlayView.topAnchor.constraint(equalTo: topAnchor),
            overlayView.rightAnchor.constraint(equalTo: rightAnchor),
            overlayView.bottomAnchor.constraint(equalTo: bottomAnchor)
        ])
        
        playbackControlView = PlaybackControlView(frame: .zero)
        playbackControlView.translatesAutoresizingMaskIntoConstraints = false
        addSubview(playbackControlView)
        NSLayoutConstraint.activate([
            playbackControlView.leftAnchor.constraint(equalTo: leftAnchor),
            playbackControlView.rightAnchor.constraint(equalTo: rightAnchor),
            playbackControlView.bottomAnchor.constraint(equalTo: bottomAnchor),
            //playbackControlView.heightAnchor.constraint(equalToConstant: 50)
        ])
        playbackControlView.backgroundColor = .gray
        
        self.playbackControlView.playbackControlDelegate = self
        self.playerView.playerControls = playbackControlView
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        
        playerView.layoutSubviews()
    }
}


extension CustomPlayerView : PlaybackControlDelegate {
    func playbackControlUpdateDuration() -> Double {
        return player?.currentItem?.playerDuration() ?? 0
    }
    
    func playbackControlUpdatePosition() -> Double {
        if let time = player?.currentTime() {
            return Double(CMTimeGetSeconds(time))
        }
        return 0
    }
    
    func playbackControlUpdateIsPlaying() -> Bool {
        if let rate = player?.rate {
            return rate != 0
        }
        return false
    }
    
    func playbackControlSeekToPosition(_ position: Double, duration: Double) {
        self.player?.seek(to: CMTime(seconds: position, preferredTimescale: 1000))
    }
    
    func playbackControlStateChanged(_ playing: Bool) {
        if playing {
            self.player?.play()
        } else {
            self.player?.pause()
        }
    }
    
    func playbackControlVisibilityWillChange(_ visible: Bool) {
    }
    
    func playbackControlVisibilityChanged(_ visible: Bool) {
        // sends the state of the native player controls to the overlay UI
        self.onPlayerControlsVisibilityChanged?(visible)
        
        setNeedsFocusUpdate()
    }
}


// view to hold AVPlayerLayer
public class PlayerView: UIControl {
    public override init(frame: CGRect) {
        super.init(frame: frame)
        setup()
    }
    
    public required init?(coder: NSCoder) {
        super.init(coder: coder)
        setup()
    }
    
    func setup() {
        self.addTarget(self, action: #selector(toggleControls), for: .touchUpInside)
    }
    
    @objc func toggleControls() {
        guard let playerControls = playerControls else { return }
        if playerControls.visible {
            playerControls.hide()
        } else {
            playerControls.show()
        }
    }
    
    var player: AVPlayer? {
        get {
            return playerLayer.player
        }
        
        set {
            playerLayer.player = newValue
        }
    }
    
    var playerLayer: AVPlayerLayer {
        return layer as! AVPlayerLayer
    }
    
    public override class var layerClass: AnyClass {
        return AVPlayerLayer.self
    }
    
    var playerControls: PlaybackControlView?
    
    public override func layoutSubviews() {
        super.layoutSubviews()
        
        updateVideoScale()
    }
    
    var scaleX: Float = 1
    var scaleY: Float = 1
    var pivotX: Float = 0
    var pivotY: Float = 0
    
    func updateVideoScale() {
        let videoBounds = playerLayer.videoRect
        
        // anchor relative to original video bounds
        let anchorX = ((CGFloat(pivotX) * videoBounds.width) + videoBounds.origin.x) / bounds.width
        let anchorY = ((CGFloat(pivotY) * videoBounds.height) + videoBounds.origin.y) / bounds.height
        
        let anchorPoint = CGPoint(x: anchorX, y: anchorY)
        layer.anchorPoint = anchorPoint

        let scaleXNonZero = scaleX != 0 ? CGFloat(scaleX) : CGFloat.leastNonzeroMagnitude
        let scaleYNonZero = scaleY != 0 ? CGFloat(scaleY) : CGFloat.leastNonzeroMagnitude
        let xPadding = 1 / scaleXNonZero * (anchorPoint.x - 0.5) * bounds.width
        let yPadding = 1 / scaleYNonZero * (anchorPoint.y - 0.5) * bounds.height
        transform = CGAffineTransform(scaleX: scaleXNonZero, y: scaleYNonZero).translatedBy(x: xPadding, y: yPadding)
    }
    
    public func setVideoScale(scaleX: Float, scaleY: Float, pivotX: Float, pivotY: Float) {
        self.scaleX = scaleX
        self.scaleY = scaleY
        self.pivotX = pivotX
        self.pivotY = pivotY
        updateVideoScale()
    }
}
