//
//  ViewController.swift
//  PlayerPluginExample
//
//  Created by Ole Helgesen on 11/03/2019.
//  Copyright © 2019 Ease Live AS. All rights reserved.
//

import UIKit
import AVFoundation
import AVKit
import EaseLiveSDK

public class MyPlayerPlugin: PlayerPluginBase {
    weak var player: AVPlayer?
    
    var changePlayerControlsVisibilityCallback: ((_ visible: Bool) -> Void)?
    var changePlayerVideoScaleCallback: ((_ scaleX: Float, _ scaleY: Float, _ pivotX: Float, _ pivotY: Float) -> Void)?
    
    public init(player: AVPlayer) {
        super.init()
        self.TAG = String(describing: MyPlayerPlugin.self)
        self.player = player
    }
    
    // Called when the overlay UI sends an event to change the video URL.
    public override func playVideoUrl(url: String) {
        if let videoUrl = URL(string: url) {
            let currentItem = AVPlayerItem(url: videoUrl)
            player?.replaceCurrentItem(with: currentItem)
            player?.play()
        }
    }
    
    // Called when the overlay UI sends an event to change the player controls visibility.
    public override func setControllerVisible(visible: Bool) {
        changePlayerControlsVisibilityCallback?(visible)
    }
    
    // Called when a swipe gesture is detected on a part of the overlay where the video is visible
    // (ie outside any EaseLive graphical element). Can be used to trigger native UI.
    public override func didSwipeStage(direction: UISwipeGestureRecognizer.Direction) {
        if direction == .down {
            print("swiped down")
        }
    }
    
    // Called when the overlay UI sends an event to seek to a player position.
    public override func setTime(time: Int64) {
        // the time parameter is the UTC timecode for the wanted seek position.
        // PlayerPluginBase#currentTimecode is UTC that was last passed to PlayerPluginBase#onTime()
        guard let currentTimecode = self.currentTimecode else {
            return
        }
        
        // seek length is the difference between the wanted UTC and the current UTC
        let diff = Float64(time - currentTimecode)
        
        if let currentPositionTime = player?.currentTime()  {
            let currentPosition = CMTimeGetSeconds(currentPositionTime)
            
            // seek position relative to the current timeline position
            let position = currentPosition + diff
            
            let seekTime: CMTime = CMTime(seconds: Double(position) / 1000, preferredTimescale: 1000)
            let tolerance: CMTime = CMTime(seconds: Double(5), preferredTimescale: 1000)
            
            player?.seek(to: seekTime, toleranceBefore: tolerance, toleranceAfter: tolerance)
        }
    }
    
    // Called when the overlay UI sends an event to set the player state.
    public override func setState(state: PlayerState) {
        if state == .paused {
            player?.pause()
        } else if state == .playing {
            player?.play()
        }
    }
    
    // Called when the overlay UI sends an event to set the playback speed
    public override func setSpeed(speed: Float) {
        player?.rate = speed
    }
    
    // Called when the overlay UI sends an event to set the audio volume
    public override func setVolume(volume: Int) {
        player?.volume = Float(volume) / Float(100)
    }
    
    // Called when the overlay UI sends an event to set the audio mute status
    public override func setMute(mute: Bool) {
        player?.isMuted = mute
    }
    
    var scaleX: Float = 1
    var scaleY: Float = 1
    var pivotX: Float = 0
    var pivotY: Float = 0
    
    func updateVideoScale() {
        self.changePlayerVideoScaleCallback?(scaleX, scaleY, pivotX, pivotY)
    }
    
    // Called when the overlay UI sends an event to change the video scale and position
    public override func setVideoScale(scaleX: Float, scaleY: Float, pivotX: Float, pivotY: Float, duration: Int) {
        self.scaleX = scaleX
        self.scaleY = scaleY
        self.pivotX = pivotX
        self.pivotY = pivotY
        UIView.animate(withDuration: TimeInterval(Double(duration) / Double(1000)), animations: {
            self.updateVideoScale()
        })
        
        // notify EaseLive that the video will scale
        onVideoScale(scaleX: scaleX, scaleY: scaleY, pivotX: pivotX, pivotY: pivotY, duration: duration)
    }
}

class ViewController: UIViewController {
    var playerPlugin: MyPlayerPlugin?
    var player: AVPlayer?
    var easeLive: EaseLive?
    
    var timedMetadataQueue = DispatchQueue(label: "TimedMetadataQueue", qos: .background)
    var timedMetadataOutput = AVPlayerItemMetadataOutput()
    
    var timeControlObserver: NSKeyValueObservation?
    var statusObserver: NSKeyValueObservation?
    var rateObserver: NSKeyValueObservation?
    var volumeObserver: NSKeyValueObservation?
    var mutedObserver: NSKeyValueObservation?
    
    // custom player controls
    var customPlayerView: CustomPlayerView?
    
    // default player controls
    var avPlayerViewController: AVPlayerViewController?
    
    // container for EaseLive overlay content
    var easeLiveView: UIView?
    
    // producer set overlay enabled, received in onEaseLiveAppStatus
    var easeLiveEnabled = true
    // user set overlay enabled, toggled by button in player controls. set false to hide overlay until enabled by user
    var easeLiveEnabledByUser = true
    
    // demo stream
    
    // VOD stream containing time in EXT-X-PROGRAM-DATE-TIME
    // let streamUrl = "https://s3-eu-west-1.amazonaws.com/vod-assets-ireland-sixty-no/dev/hls_vod_with_id3_and_programtime_nosound/stream.m3u8"
    
    // live stream containing time in ID3 tags
    let streamUrl = "https://eu-dev.stream.easelive.tv/fotball/ngrp:Stream1_all/playlist.m3u8?DVR"
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(didEnterBackground),
                                               name: UIApplication.didEnterBackgroundNotification,
                                               object: nil)
        
        NotificationCenter.default.addObserver(self,
                                               selector: #selector(willEnterForeground),
                                               name: UIApplication.willEnterForegroundNotification,
                                               object: nil)
        
        // set .mixWithOthers option of audio session if you want player and EaseLive to be able to play sound at the same time.
        // This allows video layers or watch party inside the EaseLive overlay to mix sound with your AVPlayer's sound
        do {
            let audioSession = AVAudioSession.sharedInstance()
            try audioSession.setCategory(.playback, mode: .moviePlayback, options: [.mixWithOthers])
            try audioSession.setActive(true)
            
        } catch {
            print("Setting category to AVAudioSessionCategoryPlayback failed.", error)
        }
        
        if let stream = URL(string: streamUrl) {
            // player setup
            let playerItem = AVPlayerItem(url: stream)
            let player = AVPlayer(playerItem: playerItem)
            setupPlayerObservers(player: player)
            self.player = player
            
            var useCustomPlayerView = true
            
            // use AVPlayerViewController's default player controls on iOS 16+, visionOS and tvOS
            #if os(tvOS)
            useCustomPlayerView = false
            #else
            if #available(iOS 16, visionOS 1, *) {
                useCustomPlayerView = false
            }
            #endif
            
            if useCustomPlayerView {
                // use custom player controls defined in CustomPlayerView
                setupCustomPlayerView()
            } else {
                // use default player controls of AVPlayerViewController
                setupDefaultPlayerView()
            }
            
            player.play()
            
            // EASELIVE setup
            
#if DEBUG
            // enables output of logs in the debug build
            EaseLive.setDebugging(debug: true)
#endif
            
            let playerPlugin = MyPlayerPlugin(player: player)
            
            playerPlugin.changePlayerControlsVisibilityCallback = { [weak self] visible in
                guard let self = self else { return }
                
                // change custom controls visibility
                if let playerView = self.customPlayerView {
                    if visible {
                        playerView.playbackControlView?.show()
                    } else {
                        playerView.playbackControlView?.hide()
                    }
                }
                
                // change default controls visibility
                if let avPlayerViewController {
                    if visible {
                        #if os(tvOS)
                        // no function to show player controls in AVPlayerViewController
                        // workaround to show player controls by pausing for a frame
                        if avPlayerViewController.player?.timeControlStatus == .playing {
                            avPlayerViewController.player?.pause()
                            avPlayerViewController.player?.play()
                        } else if avPlayerViewController.player?.timeControlStatus == .paused {
                            avPlayerViewController.player?.play()
                            avPlayerViewController.player?.pause()
                        }
                        #endif
                    }
                }
            }
            
            playerPlugin.changePlayerVideoScaleCallback = { [weak self] scaleX, scaleY, pivotX, pivotY in
                guard let self = self else { return }
                
                // scale video in custom player
                if let customPlayerView {
                    customPlayerView.playerView.setVideoScale(scaleX: scaleX, scaleY: scaleY, pivotX: pivotX, pivotY: pivotY)
                }
                // scale video in AVPlayerViewController
                else if let avPlayerViewController {
                    avPlayerViewController.setVideoScale(scaleX: scaleX, scaleY: scaleY, pivotX: pivotX, pivotY: pivotY)
                }
            }
            self.playerPlugin = playerPlugin
            
            // register for notifications from the SDK
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveError(notification:)),
                                                   name: EaseLiveNotificationKeys.easeLiveError,
                                                   object: nil)
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveReady(notification:)),
                                                   name: EaseLiveNotificationKeys.easeLiveReady,
                                                   object: nil)
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveAppStatus(notification:)),
                                                   name: EaseLiveNotificationKeys.bridgeAppStatus,
                                                   object: nil)
            NotificationCenter.default.addObserver(self,
                                                   selector: #selector(onEaseLiveBridgeMessage(notification:)),
                                                   name: EaseLiveNotificationKeys.bridgeMessage,
                                                   object: nil)
            
            // The overlay will be added as a subview to the view that is passed to the EaseLive constructor.
            // This view should be layered above the player video surface and below the player controls, and must be able to receive touch events.
            
            let accountId: String
            let projectId: String?
            let programId: String
            let env: String
            var params: [String: Any] = [:]
            #if os(tvOS)
                accountId = "tutorials"
                projectId = "580e5ba1-423d-4c17-9a86-08b75e88bb7e"
                programId = "cloud-capture-demo"
                env = "prod"
            #else
                accountId = "tutorials"
                projectId = "0346ae3e-7a91-4760-bcd3-cd84bb6790dd"
                programId = "2d2711ff-6ff2-41c1-a141-060e9ffa2c38"
                env = "prod"
            
                if avPlayerViewController != nil {
                    // AVPlayerViewController on iOS and visionOS:
                    // Sets parameter to enable passthrough of touches on the overlay background.
                    // This is done because AVPlayerViewController does not have functionality to toggle the player controls visibility or know the current visibility, so the touches needs to be handled by the native view.
                    params["stageTouchPassthrough"] = true
                }
            #endif
            
            self.easeLive = EaseLive(parentView: easeLiveView!,
                                     accountId: accountId,
                                     projectId: projectId,
                                     programId: programId,
                                     env: env,
                                     params: params,
                                     playerPlugin: playerPlugin)
            
            // before calling easeLive.create() you can register other plugins using easeLive.use()
            self.easeLive?.create()
            
            // a call to onReady() on the player plugin is required to let the SDK know the player is ready, so it will know to proceed with loading the overlay UI.
            // this could be used to wait to load the overlay UI until the video is loaded
            self.playerPlugin?.onReady()
        }
    }
    
    override func viewLayoutMarginsDidChange() {
        
        super.viewLayoutMarginsDidChange()
        
        if avPlayerViewController != nil {
            // reapply AVPlayerViewController videoscale after rotation
            UIView.animate(withDuration: TimeInterval(0.1), animations: {
                let l = self.avPlayerViewController?.findPlayerView(parentView: self.view)?.layer
                l?.transform = CATransform3DMakeScale(1, 1, 0)
            }) { finished in
                self.playerPlugin?.updateVideoScale()
            }
        }
    }
    
    @objc func didEnterBackground() {
        self.easeLive?.pause()
    }

    @objc func willEnterForeground() {
        self.easeLive?.load()
    }
    
    // setup a player using custom controls
    func setupCustomPlayerView() {
        let playerView = CustomPlayerView(frame: .zero)
        playerView.translatesAutoresizingMaskIntoConstraints = false
        playerView.player = player
        playerView.onPlayerControlsVisibilityChanged = { [weak self] visible in
            self?.playerPlugin?.onControllerVisibilityChanged(visible: visible)
        }
        self.customPlayerView = playerView
        view.addSubview(playerView)
        
        NSLayoutConstraint.activate([
            playerView.leftAnchor.constraint(equalTo: view.leftAnchor),
            playerView.topAnchor.constraint(equalTo: view.topAnchor),
            playerView.rightAnchor.constraint(equalTo: view.rightAnchor),
            playerView.bottomAnchor.constraint(equalTo: view.bottomAnchor)
        ])
        
        easeLiveView = playerView.overlayView
        
        playerView.playbackControlView.show()
    }
    
    // setup a player using default AVPlayerViewController controls
    func setupDefaultPlayerView() {
        
        let easeLiveView = UIView(frame: .zero)
        easeLiveView.translatesAutoresizingMaskIntoConstraints = false
        
        let avpvc = AVPlayerViewController()
        avpvc.delegate = self
        avPlayerViewController = avpvc
        avpvc.showsPlaybackControls = true
        avpvc.allowsPictureInPicturePlayback = true
        #if os(iOS)
            avpvc.canStartPictureInPictureAutomaticallyFromInline = true
        #endif
        avpvc.player = player

        #if os(tvOS)
        if #available(tvOS 15.0, *) {
            avpvc.transportBarIncludesTitleView = true
            avpvc.playbackControlsIncludeTransportBar = true
            
            // example of a button to let the user toggle the overlay visibility
            let toggleOverlayAction = UIAction(title: "Toggle overlay", image: easeLiveEnabledByUser ? .remove : .add) { [weak self] action in
                guard let self = self else { return }
                
                if easeLiveEnabled {
                    easeLiveEnabledByUser = !easeLiveEnabledByUser
                    self.easeLiveView?.isHidden = false
                    action.image = easeLiveEnabledByUser ? .remove : .add
                    
                    UIView.animate(withDuration: 0.2) {
                        self.easeLiveView?.alpha = self.easeLiveEnabledByUser ? 1 : 0
                        self.setNeedsFocusUpdate()
                    }
                }
            }

            avpvc.transportBarCustomMenuItems = [toggleOverlayAction]
        }
        #endif
        
        avpvc.view.frame = view.bounds
        avpvc.beginAppearanceTransition(true, animated: true)
        addChild(avpvc)
        view.addSubview(avpvc.view)
        avpvc.didMove(toParent: self)
        avpvc.endAppearanceTransition()
        
        #if os(tvOS)
            // NOTE: for use with AVPlayerViewController, the EaseLive parent view must be added in front of the AVPlayerViewController's view, or EaseLive cannot get focus, because the player controls cover the whole screen
            view.addSubview(easeLiveView)
            easeLiveView.superview?.bringSubviewToFront(easeLiveView)
            
            NSLayoutConstraint.activate([
                easeLiveView.leftAnchor.constraint(equalTo: view.leftAnchor),
                easeLiveView.topAnchor.constraint(equalTo: view.topAnchor),
                easeLiveView.rightAnchor.constraint(equalTo: view.rightAnchor),
                easeLiveView.bottomAnchor.constraint(equalTo: view.bottomAnchor)
            ])
        #else
            if #available(iOS 16, visionOS 1, *) {
                // iOS 16+ and visionOS, show EL in AVPlayerViewController's contentOverlayView
                if let view = avpvc.contentOverlayView {
                    view.addSubview(easeLiveView)
                    easeLiveView.leftAnchor.constraint(equalTo: view.leftAnchor).isActive = true
                    easeLiveView.topAnchor.constraint(equalTo: view.topAnchor).isActive = true
                    easeLiveView.rightAnchor.constraint(equalTo: view.rightAnchor).isActive = true
                    easeLiveView.bottomAnchor.constraint(equalTo: view.bottomAnchor).isActive = true
                }
            } else {
                // In AVPlayerViewController on iOS 15 and older contentOverlayView is not interactive, so there isn't any good solution to insert the EaseLive overlay between the video surface and the player controls of AVPlayerViewController.
                // instead use custom player controls or only show EaseLive on iOS 16 and newer
            }
        #endif
        
        self.easeLiveView = easeLiveView
        easeLiveView.isHidden = !easeLiveEnabled
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        navigationController?.isNavigationBarHidden = true
        player?.play()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        player?.pause()
        
        if isBeingDismissed || isMovingFromParent || navigationController?.isBeingDismissed ?? false {
            easeLive?.destroy()
            NotificationCenter.default.removeObserver(self)
        }
    }
    
    override var preferredFocusEnvironments: [UIFocusEnvironment] {
        if let customPlayerView, customPlayerView.playbackControlView.visible {
            print("focus custom player controls")
            return [customPlayerView.playbackControlView]
        }
        
        if let easeLiveView, let easeLive, !easeLiveView.isHidden && !easeLive.preferredFocusEnvironments.isEmpty {
            print("focus EL")
            return easeLive.preferredFocusEnvironments
        }
        
        if let avPlayerViewController {
            print("focus AVPlayerViewController")
            return avPlayerViewController.preferredFocusEnvironments
        }
        
        print("focus app")
        return super.preferredFocusEnvironments
    }
    
    // notification that the overlay successfully loaded
    @objc func onEaseLiveReady(notification: Notification) {
        print("EaseLive ready")
    }
    
    // notification that the producer changed status of the loaded overlay.
    // When the overlay was disabled, it should be removed
    @objc func onEaseLiveAppStatus(notification: Notification) {
        if let status = notification.userInfo?[EaseLiveNotificationKeys.statusUserInfoKey] as? String {
            print("EaseLive status: \(status)")
            if status == "disabled" {
                easeLiveEnabled = false
                easeLive?.destroy()
                easeLive = nil
                easeLiveView?.isHidden = true
            }
            else if status == "hidden" {
                easeLiveEnabled = false
                easeLiveView?.isHidden = true
            }
            else if status == "enabled" {
                easeLiveEnabled = true
                easeLiveView?.isHidden = false
            }
            setNeedsFocusUpdate()
            updateFocusIfNeeded()
        }
    }
    
    // notification that an error occurred
    @objc func onEaseLiveError(notification: Notification) {
        if let error = notification.userInfo?[EaseLiveNotificationKeys.errorUserInfoKey] as? EaseLiveError {
            if error.level == .fatal {
                // fatal error. For example the UI failed to load.
                // remove the overlay and fallback to a normal video
                easeLive?.destroy()
                easeLive = nil
                easeLiveView?.isHidden = true
            } else {
                // non-fatal error/warning.
            }
        }
    }
    
    // custom message from the overlay UI
    @objc func onEaseLiveBridgeMessage(notification: Notification) {
        do {
            guard let jsonString = notification.userInfo?[EaseLiveNotificationKeys.jsonStringUserInfoKey] as? String else { return }
            guard let json = try JSONSerialization.jsonObject(with: Data(jsonString.utf8)) as? [String: Any] else { return }
            guard let event = json["event"] as? String, let metadata = json["metadata"] as? [String: Any] else { return }
            
            // example of a message used for sharing a Watch Party invitation
            if event == "inviteMessage", let message = metadata["message"] as? String {
#if os(iOS)
                let objectsToShare = [message] as [Any]
                let activityVC = UIActivityViewController(activityItems: objectsToShare, applicationActivities: nil)
                activityVC.excludedActivityTypes = [.airDrop, .addToReadingList]
                self.present(activityVC, animated: true) {}
#endif
            }
        } catch {
        }
    }
}

extension AVPlayerItem {
    // get correct duration of live stream with seekable buffer
    func playerDuration() -> Float64 {
        var duration: Float64 = 0
        if status == .readyToPlay, let range = seekableTimeRanges.last {
            let rangeDuration = CMTimeGetSeconds(range.timeRangeValue.duration)
            
            if !rangeDuration.isNaN && !rangeDuration.isInfinite {
                duration = rangeDuration
            }
        }
        return duration
    }
}

// listen for changes in AVPlayer and send the change to EaseLive
extension ViewController: AVPlayerItemMetadataOutputPushDelegate {
    func setupPlayerObservers(player: AVPlayer) {
        // set up observers for change in player state and metadata
        self.statusObserver = player.observe(\.currentItem?.status, options: [.initial, .new], changeHandler: { [weak self] (player, change) in
            if let itemStatus = player.currentItem?.status {
                Task { [weak self] in
                    await self?.onStatusChanged(itemStatus: itemStatus)
                }
            }
        })
        
        self.timeControlObserver = player.observe(\.timeControlStatus, options: [.initial, .new], changeHandler: { [weak self] player, change in
            let status = player.timeControlStatus
            Task { [weak self] in
                await self?.onTimeControlStatusChanged(status: status)
            }
        })
        
        self.rateObserver = player.observe(\.rate, options: [.initial, .new], changeHandler: { [weak self] (player, change) in
            Task { [weak self] in
                await self?.onRateChanged(rate: player.rate)
            }
        })
        
        self.volumeObserver = player.observe(\.volume, changeHandler: { [weak self] player, change in
            Task { [weak self] in
                await self?.onVolumeChanged(volume: player.volume)
            }
        })
        
        self.mutedObserver = player.observe(\.isMuted) { [weak self] player, change in
            Task { [weak self] in
                await self?.onMutedChanged(isMuted: player.isMuted)
            }
        }
        
        // reads time from EXT-X-PROGRAM-DATE-TIME
        player.addPeriodicTimeObserver(forInterval: CMTime(seconds: 0.5, preferredTimescale: CMTimeScale(NSEC_PER_SEC)), queue: .main) { [weak self] (time) in
            Task { [weak self] in
                await self?.onPlayerTimeChanged()
            }
        }
        
        self.timedMetadataOutput.setDelegate(self, queue: timedMetadataQueue)
        
        player.currentItem?.add(timedMetadataOutput)
    }
    
    // sends the current audio volume to EaseLive
    private func onVolumeChanged(volume: Float) {
        self.playerPlugin?.onVolumeChanged(volume: Int(floorf(volume * 100)))
    }
    
    // sends the current audio mute status to EaseLive
    private func onMutedChanged(isMuted: Bool) {
        self.playerPlugin?.onMuteChanged(mute: isMuted)
    }
    
    // sends the current player state to EaseLive
    private func onTimeControlStatusChanged(status: AVPlayer.TimeControlStatus) {
        if status == .playing {
            self.playerPlugin?.onState(state: .playing)
        } else if status == .paused {
            self.playerPlugin?.onState(state: .paused)
        } else if status == .waitingToPlayAtSpecifiedRate {
            self.playerPlugin?.onState(state: .buffering)
        }
    }
    
    // sends the current player speed to EaseLive
    private func onRateChanged(rate: Float) {
        self.playerPlugin?.onSpeedChanged(speed: rate)
    }
    
    // send when the player is ready or received a fatal error
    private func onStatusChanged(itemStatus: AVPlayerItem.Status) {
        switch itemStatus {
        case .failed:
            self.playerPlugin?.onError(error: EaseLiveError.playerConnectivity(level: .fatal, reason: "Failed to play stream"))
        default:
            print("unknown status for player")
        }
    }
    
    // send the absolute time of the current playback position. How this is done depends on how the time is inserted in the stream
    // NOTE: pick and use only one of the methods, either ID3 or EXT-X-PROGRAM-DATE-TIME. Both are here as an example. Using multiple time sources can lead to fluctuations
    
    // Listen for when the player reads time from HLS manifest tag EXT-X-PROGRAM-DATE-TIME
    public func onPlayerTimeChanged() {
        guard let currentItem = self.player?.currentItem else { return }
        
        if let pdt = currentItem.currentDate() {
            // all values in UTC milliseconds
            let time = Int64(pdt.timeIntervalSince1970 * 1000)
            let position = Int64(CMTimeGetSeconds(currentItem.currentTime()) * 1000)
            let duration = Int64(currentItem.playerDuration() * 1000)
            self.playerPlugin?.onTime(time: time, position: position, duration: duration)
        }
    }
    
    // or read time from ID3 tags in the stream
    nonisolated public func metadataOutput(_ output: AVPlayerItemMetadataOutput, didOutputTimedMetadataGroups groups: [AVTimedMetadataGroup], from track: AVPlayerItemTrack?) {
        for group in groups {
            for item in group.items {
                if let string = item.stringValue, let utc = parseUtcFromId3Value(string) {
                    let time = utc
                    Task { [weak self] in
                        await self?.sendTime(time)
                    }
                }
            }
        }
    }
    
    func sendTime(_ time: Int64) {
        guard let player else { return }
        guard let currentItem = player.currentItem else { return }
        
        let duration = Int64(currentItem.playerDuration() * 1000)
        
        let position = duration > 0 ? Int64(CMTimeGetSeconds(player.currentTime()) * 1000) : 0
        playerPlugin?.onTime(time: time, position: position, duration: duration)
    }
    
    // Parse ID3 metadata value to retrieve absolute time for the current frame and return it as UTC milliseconds
    nonisolated func parseUtcFromId3Value(_ value: String) -> Int64? {
        var utc: Int64?
        
        // NOTE: must modify to extract the correct metadata frame from your stream.
        // this is an example of how it is read from ID3 in EaseLive's test streams only.
        
        // example: JSON with the time as milliseconds, eg. {"ut":"1557311377000"}
        if let json = try? JSONSerialization.jsonObject(with: Data(value.utf8)) as? [String: Any] {
            if let utcInt = json["ut"] as? Int64 {
                utc = utcInt
            } else if let utcString = json["ut"] as? String, let utcInt = Int64(utcString) {
                utc = utcInt
            }
        }
        
        // example: string with the time as ISO 8601, eg. 2019-05-08T10:29:37.000Z
        if utc == nil {
            let formatter = ISO8601DateFormatter()
            formatter.formatOptions = [.withInternetDateTime, .withFractionalSeconds]
            if let date = formatter.date(from: value) {
                utc = Int64(date.timeIntervalSince1970 * 1000)
            }
        }
        
        return utc
    }
}


extension ViewController: @preconcurrency AVPlayerViewControllerDelegate {
#if os(tvOS)
    // listen for AVPlayerViewController's controls visibility on tvOS
    func playerViewController(_ playerViewController: AVPlayerViewController, willTransitionToVisibilityOfTransportBar visible: Bool, with coordinator: AVPlayerViewControllerAnimationCoordinator) {
        // when player controls hide, move focus back to EaseLive
        if !visible {
            if easeLive != nil && easeLiveEnabled && easeLiveEnabledByUser {
                UIView.animate(withDuration: 0.2) {
                    self.easeLiveView?.alpha = 1
                    self.setNeedsFocusUpdate()
                }
            }
        } else {
            UIView.animate(withDuration: 0.2) {
                self.easeLiveView?.alpha = 0
                self.setNeedsFocusUpdate()
            }
        }
        
        playerPlugin?.onControllerVisibilityChanged(visible: visible)
    }
#endif
    
    func playerViewControllerWillStartPictureInPicture(_ playerViewController: AVPlayerViewController) {
        easeLiveView?.isHidden = true
    }
    
    func playerViewControllerDidStopPictureInPicture(_ playerViewController: AVPlayerViewController) {
        easeLiveView?.isHidden = !easeLiveEnabled
    }
}

extension AVPlayerViewController {
    // find the view that has the video rendering layer
    func findPlayerView(parentView: UIView) -> UIView? {
        for v in parentView.subviews {
            if v.layer as? AVPlayerLayer != nil {
                return v
            } else {
                return findPlayerView(parentView: v)
            }
        }
        return nil
    }
    
    // scale player layer
    func setVideoScale(scaleX: Float, scaleY: Float, pivotX: Float, pivotY: Float) {
        if let avpcView = self.viewIfLoaded,
           let playerView = findPlayerView(parentView: avpcView),
           let playerLayer = playerView.layer as? AVPlayerLayer {
            let videoBounds = playerLayer.videoRect
            let defaultAnchorPoint = CGPoint(x: 0.5, y: 0.5)
            let anchorPoint = CGPoint(x: CGFloat(pivotX), y: CGFloat(pivotY))
            var shiftX = CGFloat(0)
            var shiftY = CGFloat(0)
            if anchorPoint.x != defaultAnchorPoint.x {
                shiftX = videoBounds.width * (anchorPoint.x - defaultAnchorPoint.x)
                
                let scaledW = videoBounds.width * CGFloat(scaleX)
                if anchorPoint.x > 0.5 {
                    shiftX -= scaledW * 0.5
                } else {
                    shiftX += scaledW * 0.5
                }
            }
            if anchorPoint.y != defaultAnchorPoint.y {
                shiftY = videoBounds.height * (anchorPoint.y - defaultAnchorPoint.y)
                
                let scaledH = videoBounds.height * CGFloat(scaleY)
                if anchorPoint.y > 0.5 {
                    shiftY -= scaledH * 0.5
                } else {
                    shiftY += scaledH * 0.5
                }
            }
            var t = CATransform3DIdentity
            t = CATransform3DTranslate(t, shiftX, shiftY, 0)
            t = CATransform3DScale(t, CGFloat(scaleX), CGFloat(scaleY), 0)
            playerLayer.transform = t
        }
    }
}
