//
//  PlaybackControlView.swift
//  Demo
//
//  Created by Ole Helgesen on 12/03/2019.
//  Copyright © 2019 Ease Live AS. All rights reserved.
//

import UIKit

@MainActor
public protocol PlaybackControlDelegate : NSObjectProtocol {
    func playbackControlUpdateDuration() -> Double
    func playbackControlUpdatePosition() -> Double
    func playbackControlUpdateIsPlaying() -> Bool
    
    func playbackControlStateChanged(_ playing: Bool)
    func playbackControlSeekToPosition(_ position: Double, duration: Double)
    func playbackControlVisibilityWillChange(_ visible: Bool)
    func playbackControlVisibilityChanged(_ visible: Bool)
}

public class PlaybackControlView: UIControl {
    var playbackButton: UIButton?
    
    #if os(tvOS)
    var playbackScrubber: UIProgressView?
    #else
    var playbackScrubber: UISlider?
    #endif
    
    var timeLabel: UILabel?
    
    var playing = true
    
    weak var playbackControlDelegate: PlaybackControlDelegate?
    
    var hideTask: DispatchWorkItem?
    var timer: Timer?
    public var visible = false
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setup()
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        setup()
    }
    
    public override func didMoveToSuperview() {
        super.didMoveToSuperview()
        self.visible = false
        self.isHidden = true
        hideTransform()
    }
    
    public override func layoutMarginsDidChange() {
        super.layoutMarginsDidChange()
        
        if (!visible) {
            hideTransform()
        }
    }
    
    public override var canBecomeFocused: Bool {
        return true
    }
    
    public override func shouldUpdateFocus(in context: UIFocusUpdateContext) -> Bool {
        if context.nextFocusedView == self {
            return true
        } else {
            hide()
            return super.shouldUpdateFocus(in: context)
        }
    }
    
    public override func pressesEnded(_ presses: Set<UIPress>, with event: UIPressesEvent?) {
        presses.forEach { press in
            if press.type == .playPause {
                playbackButtonPressed()
            }
            if press.type == .select {
                playbackButtonPressed()
            }
            
            if !playing {
                if press.type == .leftArrow {
                    changeSeekPosition(-10)
                }
                if press.type == .rightArrow {
                    changeSeekPosition(10)
                }
            }
        }
    }
    
    func changeSeekPosition(_ diff: Double) {
        guard duration > 0 else { return }
        
        var from = position
        if let seekPosition {
            from = seekPosition
        }
        
        let p = max(0, min(duration, (from + diff)))
        self.seekPosition = p
        #if os(tvOS)
        playbackScrubber?.progress = duration == 0 ? 0 : Float(p / duration)
        #endif
        self.timeLabel?.text = self.formattedTime(p)
    }
    
    func setup() -> Void {
        let timeLabel = UILabel()
        timeLabel.font = UIFont.preferredFont(forTextStyle: .body)
        self.timeLabel = timeLabel
        
        #if os(tvOS)
        backgroundColor = UIColor.systemGray
        addTarget(self, action: #selector(playbackButtonPressed), for: .primaryActionTriggered)

        let panGesture = UIPanGestureRecognizer(target: self, action: #selector(didPan(_:)))
        addGestureRecognizer(panGesture)
        
        let playbackScrubber = UIProgressView()
        #else
        let playbackButton = UIButton(type: .system)
        if #available(iOS 15.0, *) {
            playbackButton.configuration?.contentInsets = NSDirectionalEdgeInsets(top: 5, leading: 10, bottom: 5, trailing: 10)
        } else {
            playbackButton.contentEdgeInsets = UIEdgeInsets(top: 5, left: 10, bottom: 5, right: 10)
        }
        
        playbackButton.accessibilityIdentifier = "playbackButton"
        playbackButton.addTarget(self, action: #selector(playbackButtonPressed), for: .touchUpInside)
        
        playbackButton.isEnabled = true
        playbackButton.isUserInteractionEnabled = true
        playbackButton.translatesAutoresizingMaskIntoConstraints = false
        playbackButton.setImage(UIImage(systemName: "play.fill"), for: UIControl.State.normal)
        self.playbackButton = playbackButton
        
        let playbackScrubber = UISlider()
        playbackScrubber.minimumValue = 0.0
        playbackScrubber.maximumValue = 0.0
        playbackScrubber.isContinuous = false
        playbackScrubber.addTarget(self, action: #selector(sliderValueDidChange(_:)), for: .valueChanged)
        playbackScrubber.isEnabled = true
        #endif
        playbackScrubber.isUserInteractionEnabled = true
        playbackScrubber.translatesAutoresizingMaskIntoConstraints = false
        self.playbackScrubber = playbackScrubber
        
        let stackView = UIStackView(frame: .zero)
        stackView.distribution = .fill
        stackView.alignment = .center
        stackView.axis = .horizontal
        if let playbackButton = self.playbackButton {
            stackView.addArrangedSubview(playbackButton)
        }
        stackView.addArrangedSubview(timeLabel)
        stackView.addArrangedSubview(playbackScrubber)
        
        var height: CGFloat = 50
        #if os(tvOS)
        height = 80
        stackView.spacing = 28.0
        #else
        stackView.spacing = 8.0
        #endif
        stackView.translatesAutoresizingMaskIntoConstraints = false
        
        addSubview(stackView)
        
        NSLayoutConstraint.activate([
            stackView.leftAnchor.constraint(equalTo: safeAreaLayoutGuide.leftAnchor, constant: 10),
            stackView.topAnchor.constraint(equalTo: safeAreaLayoutGuide.topAnchor),
            stackView.bottomAnchor.constraint(equalTo: safeAreaLayoutGuide.bottomAnchor),
            stackView.rightAnchor.constraint(equalTo: safeAreaLayoutGuide.rightAnchor, constant: -10),
            stackView.heightAnchor.constraint(equalToConstant: height)
        ])
        
        hideTransform()
        
        timer = Timer.scheduledTimer(timeInterval: 1, target: self, selector: #selector(fireTimer), userInfo: nil, repeats: true)
    }
    
    var panPos: Double?
    var panStartX: Double?
    
    @objc func didPan(_ sender: UIPanGestureRecognizer) {
        #if os(tvOS)
        guard duration > 0, !playing else { return }
        
        let translation = sender.translation(in: self.playbackScrubber)
        if abs(translation.y) > abs(translation.x) {
            return
        }
        
        let translationX = translation.x
        
        switch sender.state {
        case .began:
            panPos = seekPosition ?? position
            
            panStartX = Double(playbackScrubber?.frame.width ?? 0) * Double(playbackScrubber?.progress ?? 0)
            
            print("pan start \(translationX)")
        case .changed:
            print("pan change \(translationX)")
            
            let centerX = (panStartX ?? 0) + translationX / 5
            let percent = centerX / Double(playbackScrubber?.frame.width ?? 0)
            let pos = percent * duration
            
            playbackScrubber?.progress = Float(percent)
            self.timeLabel?.text = self.formattedTime(pos)
            self.seekPosition = Double(playbackScrubber?.progress ?? 0) * duration
            
            /*
            let pos = max(0, min((panPos ?? 0) + translationX, duration))
            let p = pos / duration
            playbackScrubber?.progress = Float(p)
            self.timeLabel?.text = self.formattedTime(pos)
            */
        case .ended, .cancelled:
            print("pan end \(translationX)")
            //changeSeekPosition(translationX)
            
        default:
            break
        }
        #endif
    }
    
    @objc func fireTimer() {
        if let d = playbackControlDelegate {
            let position = d.playbackControlUpdatePosition()
            let duration = d.playbackControlUpdateDuration()
            let isPlaying = d.playbackControlUpdateIsPlaying()
            
            setPlaying(playing: isPlaying)
            setPosition(position: position, duration: duration)
        }
    }
    
    @objc func playbackButtonPressed () {
        let playing = !self.playing
        setPlaying(playing: playing)
        
        self.playbackControlDelegate?.playbackControlStateChanged(playing)
        
        self.hideDelayed()
    }
    
    #if !os(tvOS)
    @objc func sliderValueDidChange(_ sender: UISlider!) {
        let position = Double(sender.value)
        let duration = Double(sender.maximumValue)
        
        self.playbackControlDelegate?.playbackControlSeekToPosition(position, duration: duration)
        
        self.hideDelayed()
    }
    #endif
    
    fileprivate func setPlaying(playing: Bool) -> Void {
        self.playing = playing
        DispatchQueue.main.async {
            if playing {
                self.playbackButton?.setImage(UIImage(systemName: "pause.fill"), for: UIControl.State.normal)
            } else {
                self.playbackButton?.setImage(UIImage(systemName: "play.fill"), for: UIControl.State.normal)
            }
            
            #if os(tvOS)
            self.playbackScrubber?.layer.borderColor = playing ? UIColor.clear.cgColor : UIColor(white: 1, alpha: 0.6).cgColor
            self.playbackScrubber?.layer.borderWidth = playing ? 0 : 1
            #endif
            
            if !playing {
                self.show()
            } else {
                if let seekPosition = self.seekPosition {
                    let newPosition = max(0, min(self.duration, (seekPosition)))
                    self.seekPosition = nil
                    self.playbackControlDelegate?.playbackControlSeekToPosition(newPosition, duration: self.duration)
                    self.hideDelayed()
                }
            }
        }
    }
    
    var position: Double = 0
    var duration: Double = 0
    var seekPosition: Double?
    
    fileprivate func setPosition(position: Double, duration: Double) -> Void {
        self.position = position
        self.duration = duration
        DispatchQueue.main.async {
            if let playbackScrubber = self.playbackScrubber {
                
                playbackScrubber.alpha = duration > 0 ? 1 : 0
#if os(tvOS)
                    if self.playing {
                        self.timeLabel?.text = self.formattedTime(position)
                        playbackScrubber.progress = duration == 0 ? 0 : Float(position / duration)
                    }
#else
                    playbackScrubber.minimumValue = 0
                    playbackScrubber.maximumValue = Float(duration)
                    
                    if(!playbackScrubber.isTracking) {
                        self.timeLabel?.text = self.formattedTime(position)
                        playbackScrubber.value = Float(min(position, duration))
                    }
#endif
            }
        }
    }
    
    public func hideTransform() {
        let b: CGFloat = self.superview?.layoutMargins.bottom ?? 0
        self.transform = CGAffineTransform(translationX: 0, y: self.frame.size.height + b + 80)
    }
    
    public func hide() {
        guard visible else {
            return
        }
        self.playbackControlDelegate?.playbackControlVisibilityWillChange(false)
        self.hideTask?.cancel()
        visible = false
        
        UIView.animate(withDuration: 0.5, animations: {
            self.hideTransform()
        }) { (finished) in
            if !self.visible {
                self.isHidden = true
            }
            self.playbackControlDelegate?.playbackControlVisibilityChanged(false)
        }
    }
    
    public func show() {
        guard !visible else {
            self.hideDelayed()
            return
        }
        
        self.playbackControlDelegate?.playbackControlVisibilityWillChange(true)
        visible = true
        self.isHidden = false
        UIView.animate(withDuration: 0.3, animations: {
            self.transform = CGAffineTransform.identity
        }) { (finished) in
            self.playbackControlDelegate?.playbackControlVisibilityChanged(true)
            self.hideDelayed()
        }
    }
    
    private func hideDelayed() {
        self.hideTask?.cancel()
        
        let task = DispatchWorkItem { [weak self] in
            var isTracking = false
            #if !os(tvOS)
                isTracking = self?.playbackScrubber?.isTracking ?? false
            #endif
            guard let hideTask = self?.hideTask, !hideTask.isCancelled && self?.playing ?? false && !isTracking else {
                return
            }
            self?.hide()
        }
        self.hideTask = task
        DispatchQueue.main.asyncAfter(deadline: .now() + 5.0, execute: task)
    }
    
    public override func willMove(toSuperview newSuperview: UIView?) {
        super.willMove(toSuperview: newSuperview)
        
        self.hideDelayed()
    }
    
    func formattedTime(_ pos: Double) -> String {
        if duration > 0 {
            return "\(formatSeconds(pos)) / \(formatSeconds(duration))"
        }
        return "LIVE"
    }
    
    func formatSeconds(_ time: Double) -> String {
        let formatter = DateComponentsFormatter()
        formatter.allowedUnits = [.hour, .minute, .second]
        formatter.unitsStyle = .positional
        formatter.zeroFormattingBehavior = .pad
        return formatter.string(from: TimeInterval(time)) ?? ""
    }
}
