//
//  ViewController.swift
//  TvAppExample
//
//  Created by Ole Helgesen on 24/06/2022.
//

import UIKit
import AVKit
import EaseLiveSDK

class ViewController: AVPlayerViewController {
    // IDs identifying the current content being played on the TV
    // these will be transfered to the phone when it starts syncing, so that it can load EaseLive for the correct program
    
    // EaseLive account ID
    let accountId = "demo"
    
    // Ease Live program ID for the current content playing on the TV
    let programId = "6ca041a9-7071-49f2-b6ba-9d3583e547cb"
    
    var playerSync: PlayerSync?
    
    var timedMetadataQueue = DispatchQueue(label: "TimedMetadataQueue", qos: .background)
    var timedMetadataOutput: AVPlayerItemMetadataOutput?
    var timeControlObserver: NSKeyValueObservation?
    var rateObserver: NSKeyValueObservation?
    var playerUpdateTimer: Timer?
    
    let timeLabel = UILabel()
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        // for the example, outputs the current absolute time read from the stream, so that it can be compared when looking on the synced phone app
        timeLabel.shadowColor = .black
        timeLabel.shadowOffset = CGSize(width: 2, height: 2)
        timeLabel.frame = CGRect(x: view.bounds.width-500, y: 0, width: 500, height: 200)
        timeLabel.autoresizingMask = [.flexibleLeftMargin, .flexibleBottomMargin]
        timeLabel.text = ""
        self.contentOverlayView?.addSubview(timeLabel)
        
        let streamUrl = URL(string: "https://eu-dev.stream.easelive.tv/fotball/ngrp:Stream1_all/playlist.m3u8?DVR")
        let playerItem = AVPlayerItem(url: streamUrl!)
        let player = AVPlayer(playerItem: playerItem)
        self.player = player
        
        // setup reading of the player state and time
        
        self.timeControlObserver = player.observe(\.timeControlStatus, options: [.initial, .new], changeHandler: { [weak self] (player, change) in
            guard let self = self else { return }
            self.onTimeControlStatusChanged(status: player.timeControlStatus)
        })
        
        self.rateObserver = player.observe(\.rate, options: [.initial, .new], changeHandler: { [weak self] (player, change) in
            guard let self = self else { return }
            self.onRateChanged(rate: player.rate)
        })
        
        let timedMetadataOutput = AVPlayerItemMetadataOutput()
        timedMetadataOutput.setDelegate(self, queue: timedMetadataQueue)
        self.timedMetadataOutput = timedMetadataOutput
        playerItem.add(timedMetadataOutput)

        // Create an action to connect phone app as second screen
        let secondScreenAction = UIAction(title: "Show stats on phone") { [weak self] action in
            guard let self = self else { return }
            self.startSecondScreen()
        }
        
        let easeLiveMenu = UIMenu(
            title: "Ease Live",
            image: UIImage(imageLiteralResourceName: "easelivelogo"),
            children: [secondScreenAction]
        )
        
        transportBarCustomMenuItems = [easeLiveMenu]
        
        player.play()
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        
        if isBeingDismissed {
            playerSync?.setState(.stopped)
            playerSync?.stop()
        }
    }
    
    func onTimeControlStatusChanged(status: AVPlayer.TimeControlStatus) {
        if status == .playing {
            playerSync?.setState(.playing)
        } else if status == .paused {
            playerSync?.setState(.paused)
        }
    }

    func onRateChanged(rate: Float) {
        playerSync?.setSpeed(rate)
    }

    func startSecondScreen() {
        if playerSync == nil {
            playerSync = PlayerSync(accountId: accountId)
        }
        
        // acquires a sync ID from the server, and starts publishing the player state
        playerSync?.start({ [weak self] result in
            guard let self = self else { return }
            
            switch result {
            case .success(let syncId):
                // transfer the sync ID and program ID from the TV to the second screen device, so that it can load EaseLive with correct content.
                // For the example it is done by showing a QR on the TV, which can be scanned on the mobile to open the mobile app.
                // A more seamless transfer could be accomplished by sending it in a push notification to the mobile app
                
                // generates app link for the mobile app, based on URL set in project settings
                self.playerSync?.createSecondScreenAppLink(
                    projectId: nil,
                    programId: self.programId,
                    env: nil,
                    completion: { [weak self] result in
                    guard let self = self else { return }
                    if let link = try? result.get() {
                        self.presentQrCode(link: link)
                    }
                })
            case .failure(let error):
                print("error acquiring sync ID", error)
            }
        })
    }
    
    func presentQrCode(link: String) {
        if let qr = UIStoryboard(name: "Main", bundle: nil).instantiateViewController(withIdentifier: "QrCodeViewController") as? QrCodeViewController {
            qr.string = link
            qr.modalPresentationStyle = .overCurrentContext
            
            // delay a bit because the player controls menu has to dismiss before presenting
            DispatchQueue.main.asyncAfter(deadline: .now() + 0.5) {
                self.present(qr, animated: true)
            }
        }
    }
}

extension ViewController : AVPlayerItemMetadataOutputPushDelegate {
    public func metadataOutput(_ output: AVPlayerItemMetadataOutput, didOutputTimedMetadataGroups groups: [AVTimedMetadataGroup], from track: AVPlayerItemTrack?) {
        for group in groups {
            for item in group.items {
                if let string = item.stringValue {
                    if let utc = parseJson(value: string) {
                        playerSync?.setTime(utc)
                        
                        DispatchQueue.main.async {
                            let d = Date(timeIntervalSince1970: Double(utc)/1000)
                            self.timeLabel.text = d.formatted(.iso8601.time(includingFractionalSeconds: true))
                        }
                    }
                }
            }
        }
    }
    
    // Parse ID3 tag content and notify EaseLive if the absolute time is found in the ID3
    func parseJson(value: String) -> Int64? {
        var utc: Int64?
        do {
            // parse the JSON data from the ID3 value
            guard let json = try JSONSerialization.jsonObject(with: Data(value.utf8), options: .fragmentsAllowed) as? [String: Any] else {
                return nil
            }
            
            // extract the UTC timecode from the JSON
            
            if let utcInt = json["utc"] as? Int64 {
                utc = utcInt
            } else if let utcInt = json["ut"] as? Int64 {
                utc = utcInt
            } else if let utcString = json["utc"] as? String, let utcInt = Int64(utcString) {
                utc = utcInt
            } else if let utcString = json["ut"] as? String, let utcInt = Int64(utcString) {
                utc = utcInt
            }
        } catch {
            
        }
        
        return utc
    }
}

// presents a QR on the TV, which will by scanned by the phone app to start syncing
class QrCodeViewController: UIViewController {
    var string: String?
    @IBOutlet weak var imageView: UIImageView!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        if let string = string {
            imageView.image = generateQRCode(from: string)
        }
    }
    
    @IBAction func clickedDone(_ sender: Any) {
        self.dismiss(animated: true)
    }
    
    func generateQRCode(from string: String) -> UIImage? {
        let data = string.data(using: String.Encoding.ascii)

        if let filter = CIFilter(name: "CIQRCodeGenerator") {
            filter.setValue(data, forKey: "inputMessage")
            let transform = CGAffineTransform(scaleX: 3, y: 3)

            if let output = filter.outputImage?.transformed(by: transform) {
                return UIImage(ciImage: output)
            }
        }

        return nil
    }
}
